package com.example.sefinsa_app;

import static com.example.sefinsa_app.api.API.urlPagos;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.ExistingWorkPolicy;
import androidx.work.ListenableWorker;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.migrations.DatabaseHelper;
import com.example.sefinsa_app.models.Pago;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

public class PagosWorker extends Worker {
    private SharedPreferences sesion;
    private final ConnectivityManager connectivityManager;
    public static final int BATCH_SIZE_PAGOS = 999999;
    public static int currentPagePagos = 0;
    public static boolean allDataLoadedPagos = false;
    public static List<Integer> allPagos = new ArrayList<>();
    public static boolean isTaskPagosCompleted = false;

    public PagosWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);
        connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
    }

    // Aquí se define el AsyncTask dentro de la misma clase
    private class SyncTask extends AsyncTask<Void, Void, Void> {
        private Context context;
        private SQLiteDatabase db;
        private List<Pago> serverPagos;
        private List<Pago> localPagos;

        // Constructor
        public SyncTask(Context context, List<Pago> serverPagos, List<Pago> localPagos, SQLiteDatabase db) {
            this.context = context;
            this.db = db;
            this.serverPagos = serverPagos;
            this.localPagos = localPagos;
        }

        // Tarea en segundo plano (ejecutada fuera del hilo principal)
        @Override
        protected Void doInBackground(Void... voids) {
            // Llamamos al método de actualización de pagos
            updatePagoInSQLite(context, serverPagos, localPagos, db);
            return null;
        }

        // Tarea que se ejecuta cuando el AsyncTask termina
        @Override
        protected void onPostExecute(Void result) {
            super.onPostExecute(result);
            isTaskPagosCompleted = true;
            // Aquí podrías actualizar la UI si es necesario
            //Log.d("Sync", "La sincronización de pagos ha terminado.");
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.N)
    @NonNull
    @Override
    public ListenableWorker.Result doWork() {
        SQLiteDatabase db = null; // Declarar fuera para asegurarte de cerrarla después
        try {

            // Verificar conexión a Internet
            if (!isNetworkAvailable(getApplicationContext())) {
                Log.d("PagosWorker", "No internet connection, retrying later.");
                return ListenableWorker.Result.retry();
            }

            // Verificar si la hora actual está dentro del rango permitido
            if (!isWithinAllowedTime()) {
                Log.d("PagosWorker", "Current time is outside allowed range, retrying later.");
                return ListenableWorker.Result.retry();
            }

            // Abrir la base de datos al inicio
            DatabaseHelper dbHelper = new DatabaseHelper(getApplicationContext());
            db = dbHelper.getWritableDatabase();

            // Realizar las operaciones necesarias
            if (!allDataLoadedPagos) {
                Log.d("PagosWorker", "Entro en loadAllData PAgos.................................");
                loadAllData(db); // Pasar la conexión como argumento
            } else {
                Log.d("PagosWorker", "Entro en checkForUpdates de PAgos.................................");
                checkForUpdates(getApplicationContext(), db); // Pasar la conexión como argumento
            }
            reprogramarWorker();
            return ListenableWorker.Result.success();

        } catch (Exception e) {
            Log.e("PagosWorker", "Error in doWork", e);
            return ListenableWorker.Result.failure();
        } finally {
            if (isTaskPagosCompleted && db != null && db.isOpen()) {
                db.close();
                Log.d("PagosWorker", "Cerrando conexión a la base de datos EN PAGOS.");
            }
        }
    }
    private void reprogramarWorker() {
        WorkManager.getInstance(getApplicationContext()).enqueueUniqueWork(
                "PagosWorker",
                ExistingWorkPolicy.REPLACE, // Evita que se creen trabajos duplicados
                new OneTimeWorkRequest.Builder(PagosWorker.class)
                        .setInitialDelay(15, TimeUnit.MINUTES)
                        .build()
        );

        Log.d("ClientesWorker", "Worker PagosWorker reprogramado para ejecutarse en 11 minuto.");
    }

    private void loadAllData(SQLiteDatabase db) {
        Context context = getApplicationContext();

        if (context == null) {
            //Log.e("PagosWorker", "Contexto es nulo. No se puede cargar la data.");
            return;
        }

        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);

        // Preparar el objeto JSON con los parámetros
        JSONObject data = new JSONObject();
        try {
            data.put("func", "index_app_Pagos");
            // Convertir allPagos a JSONArray
            JSONArray pagosArray = new JSONArray();
            for (Integer pagoId : allPagos) {
                pagosArray.put(pagoId);
            }

            // Añadir la lista de pagos al objeto JSON
            data.put("PrestamosId", pagosArray);

            Log.d("PAGOS", "Parámetros enviados WORKER Pagos: " + data);
        } catch (JSONException e) {
            e.printStackTrace();
            return; // Salir si hay error al construir el objeto JSON
        }

        // Enviar la solicitud con Volley
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, urlPagos, data,
                response -> {
                        try {
                            JSONArray data1 = response.getJSONArray("data");
                            List<Pago> batchPagos = new ArrayList<>();

                            // Procesar los datos recibidos y convertirlos en objetos Pago
                            for (int i = 0; i < data1.length(); i++) {
                                JSONObject obj = data1.getJSONObject(i);
                                Gson gson = new Gson();
                                Pago pago = gson.fromJson(obj.toString(), Pago.class);
                                batchPagos.add(pago);
                            }

                            // Insertar la lista de pagos en SQLite
                            insertPagosIntoSQLiteAllData(context, batchPagos, db);

                            // Verificar si quedan registros pendientes de descarga
                            if (batchPagos.size() < BATCH_SIZE_PAGOS) {
                                allDataLoadedPagos = true;
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        } finally {
                            allDataLoadedPagos = true;
                        }
                },
                error -> {
                    Log.e("PAGOS", "Error de red Pagos: " + error.toString());

                    // Verificar si el error incluye una respuesta
                    if (error.networkResponse != null && error.networkResponse.data != null) {
                        String responseBody = new String(error.networkResponse.data);
                        //Log.e("PAGOS", "Respuesta de error del servidor: " + responseBody);
                    } else {
                        Log.e("PAGOS", "No se recibió respuesta del servidor.");
                    }

                    handleNetworkError(context);
                }

        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void insertPagosIntoSQLiteAllData(Context context, List<Pago> pagos, SQLiteDatabase db) {
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }

        Log.d("SQLite", "Iniciando inserción masiva en SQLite...");
        final int BATCH_SIZE = 500;

        // Inicia la transacción antes de procesar los lotes
        db.beginTransaction();
        try {
            for (int i = 0; i < pagos.size(); i += BATCH_SIZE) {
                int end = Math.min(i + BATCH_SIZE, pagos.size());
                List<Pago> batch = pagos.subList(i, end);

                // Procesa cada lote de registros
                for (Pago pago : batch) {
                    ContentValues values = new ContentValues();
                    values.put("id", pago.getId());
                    values.put("prestamo_id", pago.getPrestamo_id());
                    values.put("cantidad_esperada_pago", pago.getCantidad_esperada_pago());
                    values.put("cantidad_normal_pagada", pago.getCantidad_normal_pagada());
                    values.put("cantidad_multa", pago.getCantidad_multa());
                    values.put("cantidad_pendiente", pago.getCantidad_pendiente());
                    values.put("cantidad_total_pagada", pago.getCantidad_total_pagada());
                    values.put("concepto", pago.getConcepto());
                    values.put("fecha_pago", pago.getFecha_pago());
                    values.put("fecha_pago_realizada", pago.getFecha_pago_realizada());
                    values.put("folio", pago.getFolio());
                    values.put("semana", pago.getSemana());
                    values.put("balance", pago.getBalance());
                    values.put("empleado_id", pago.getEmpleado_id() != null ? pago.getEmpleado_id() : "NULL");
                    values.put("status", pago.getStatus());
                    values.put("updated_at", pago.getUpdated_at());
                    values.put("tipo_pago", pago.getTipo_pago());
                    values.put("recuperado", pago.getRecuperado());

                    long newRowId = db.insertWithOnConflict("pagos", null, values, SQLiteDatabase.CONFLICT_REPLACE);

                    // Verificación de inserción
                    if (newRowId == -1) {
                        //Log.e("SQLite", "Error al insertar el Pago con ID: " + pago.getId());
                    }
                }
            }

            // Marca la transacción como exitosa después de procesar todos los lotes
            db.setTransactionSuccessful();
            //Log.d("SQLite", "Inserción masiva completada con éxito.");
        } catch (Exception e) {
            Log.e("SQLite", "Error durante la inserción masiva: " + e.getMessage());
        } finally {
            db.endTransaction();
            isTaskPagosCompleted=true;
        }
    }

    private void insertPagosIntoSQLite(Context context, List<Pago> pagos, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        //Log.e("SQLite", "Entro a la función de insertPagosIntoSQLite()............................");
        // Usar una transacción para mejorar la eficiencia
            try {
                for (Pago pago : pagos) {
                    ContentValues values = new ContentValues();
                    values.put("id", pago.getId());
                    values.put("prestamo_id", pago.getPrestamo_id());
                    values.put("cantidad_esperada_pago", pago.getCantidad_esperada_pago());
                    values.put("cantidad_normal_pagada", pago.getCantidad_normal_pagada());
                    values.put("cantidad_multa", pago.getCantidad_multa());
                    values.put("cantidad_pendiente", pago.getCantidad_pendiente());
                    values.put("cantidad_total_pagada", pago.getCantidad_total_pagada());
                    values.put("concepto", pago.getConcepto());
                    values.put("fecha_pago", pago.getFecha_pago());
                    values.put("fecha_pago_realizada", pago.getFecha_pago_realizada());
                    values.put("folio", pago.getFolio());
                    values.put("semana", pago.getSemana());
                    values.put("balance", pago.getBalance());
                    values.put("empleado_id", pago.getEmpleado_id() != null ? pago.getEmpleado_id() : "NULL");
                    values.put("status", pago.getStatus());
                    values.put("updated_at", pago.getUpdated_at());
                    values.put("tipo_pago", pago.getTipo_pago());
                    values.put("recuperado", pago.getRecuperado());

                    long newRowId = db.insertWithOnConflict("pagos", null, values, SQLiteDatabase.CONFLICT_REPLACE);
                    remainingComparisons.decrementAndGet();
                    // Verificación de inserción
                    if (newRowId == -1) {
                        //Log.e("SQLite", "Error al insertar el Pago con ID: " + pago.getId());
                    }
                }
            } catch (Exception e) {
                Log.e("SQLite", "Error durante la inserción en lote: " + e.getMessage());
            }
    }

    private void handleNetworkError(Context context) {
        if (isNetworkAvailable(context)) {
            Log.d("PagosWorker", "Network available, resuming work.");
            enqueueWork(context); // Siempre intenta encolar el trabajo si hay red disponible
        } else {
            Log.d("PagosWorker", "No network available, stopping work.");
            cancelAndRescheduleWork(context);
        }
    }
    private void cancelAndRescheduleWork(Context context) {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();

        OneTimeWorkRequest retryWorkRequest = new OneTimeWorkRequest.Builder(PagosWorker.class)
                .setConstraints(constraints)
                .setInitialDelay(15, TimeUnit.MINUTES) // Reprogramar después de 5 minutos
                .build();

        WorkManager.getInstance(context).enqueue(retryWorkRequest);
        Log.d("PagosWorker", "Reprogramado trabajo debido a error.");
    }
    private boolean isWithinAllowedTime() {
        Calendar calendar = Calendar.getInstance();
        int hourOfDay = calendar.get(Calendar.HOUR_OF_DAY);
        return hourOfDay >= 7 && hourOfDay <= 22;
    }

    private boolean isNetworkAvailable(Context context) {
        try {
            NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
            return activeNetworkInfo != null && activeNetworkInfo.isConnected();
        } catch (Exception e) {
            Log.e("PagosWorker", "Error checking network availability", e);
            return false;
        }
    }

    public static void enqueueWork(Context context) {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED) // Requiere conexión a Internet
                .build();

        // Crear un OneTimeWorkRequest para ejecutar el trabajo una sola vez
        OneTimeWorkRequest oneTimeWorkRequest = new OneTimeWorkRequest.Builder(PagosWorker.class)
                .setConstraints(constraints) // Agregar las restricciones definidas
                //.setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 120, TimeUnit.MINUTES) // Configuración de reintentos
                .build();

        // Encolar el trabajo como único
        WorkManager.getInstance(context.getApplicationContext()).enqueueUniqueWork(
                "PagosWorker", // Nombre único para identificar el trabajo
                ExistingWorkPolicy.REPLACE, // Reemplazar si ya existe un trabajo encolado con este nombre
                oneTimeWorkRequest
        );

        Log.d("PagosWorker", "Se ha encolado un OneTimeWorkRequest para PagosWorker.");
    }

    private void insertDatosLocalAlServidor(final Context context, final Map<String, String> pagoLocal, SQLiteDatabase db, final Runnable onComplete) {
        Log.d("PrestamoWorker", "Sending insertDatosLocalAlServidor local data to the server PRESTAMOS...");

        Handler mainHandler = new Handler(Looper.getMainLooper());
        if (context == null) {
            Log.e("PrestamosWorker", "Context is null. Cannot send data to server.");
            if (onComplete != null) onComplete.run(); // Llamar a onComplete incluso si hay un error
            return; // Salir si el contexto es nulo
        }
        String prestamoIdLocalOK = pagoLocal.get("id");
        JSONObject data = new JSONObject();
        try {
            // Obtén los valores del mapa en lugar de usar métodos de la clase `Prestamo`
            data.put("func", "updateFromLocal");
            data.put("prestamo_id", pagoLocal.get("prestamo_id"));
            data.put("semana", pagoLocal.get("semana"));
            data.put("tipo_pago", pagoLocal.get("tipo_pago"));
            data.put("statusPago", pagoLocal.get("status"));
            data.put("empleado_id", pagoLocal.get("empleado_id"));
            data.put("concepto", pagoLocal.get("concepto"));
            data.put("updated_at", pagoLocal.get("updated_at"));
            data.put("fecha_pago_realizada", pagoLocal.get("fecha_pago_realizada"));
            data.put("cantidad_normal_pagada", pagoLocal.get("cantidad_normal_pagada"));
            data.put("cantidad_multa", pagoLocal.get("cantidad_multa"));
            data.put("folio", pagoLocal.get("folio"));

            Log.d("PrestamoWorker", "Parametros enviados al servidor: " + data);
        } catch (JSONException e) {
            Log.e("PrestamoWorker", "Error al crear JSON para el préstamo: " + e.getMessage(), e);
            if (onComplete != null) onComplete.run(); // Llamar a onComplete incluso si hay un error
            return; // Si hay un error en la creación del JSON, salir
        }

        // Realizar la solicitud al servidor
        Log.d("PrestamoWorker", "Realizando la solicitud POST al servidor: " + API.urlPagos);
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPagos, data,
                response -> {
                    Log.d("PrestamoWorker", "Datos locales enviados y respuesta recibida PRESTAMOS: " + response.toString());

                    try {
                        // Acceder al objeto "data" en la respuesta
                        JSONObject data2 = response.getJSONObject("data");

                        // Ahora obtener el "prestamo_id" desde el objeto "data"
                        String pagoIdServidor = data2.getString("pago_id");

                        Log.d("PrestamoWorker", "ID del préstamo recibido del servidor: " + pagoIdServidor);

                        // Si la respuesta es exitosa y hay un ID, actualiza el préstamo en SQLite
                        if (pagoIdServidor != null && !pagoIdServidor.isEmpty()) {
                            String prestamoIdLocal = pagoLocal.get("id");
                            Log.d("PrestamoWorker", "Actualizando el préstamo en SQLite con el ID: " + pagoIdServidor);
                            updatePagoIdInSQLite(context, prestamoIdLocal, pagoIdServidor, db);
                        } else {
                            Log.e("PrestamoWorker", "El servidor no ha devuelto un prestamo_id válido.");
                        }
                    } catch (JSONException e) {
                        Log.e("PrestamoWorker", "Error al procesar la respuesta del servidor: " + e.getMessage(), e);
                    } finally {
                        // Llamar al callback para notificar que esta tarea ha terminado
                        if (onComplete != null) onComplete.run();
                    }
                },
                error -> {
                    mainHandler.post(() -> {
                        try {
                            // Mensaje general del error
                            String errorMessage = error.getMessage();
                            //Log.e("PrestamoWorker", "ERROR al enviar datos locales pagos: " + errorMessage, error);

                            // Verificar si hay una respuesta de red
                            if (error.networkResponse != null) {
                                Log.e("PrestamoWorker", "Código de respuesta del error: " + error.networkResponse.statusCode);

                                // Procesar el cuerpo de la respuesta si está presente
                                String responseBody = new String(error.networkResponse.data, "UTF-8");
                                JSONObject jsonObject = new JSONObject(responseBody);

                                // Extraer el mensaje del servidor
                                if (jsonObject.has("message")) {
                                    String serverMessage = jsonObject.getString("message");
                                    Log.e("PrestamoWorker", "Mensaje del servidor: " + serverMessage);

                                    // Verificar si el mensaje contiene "semana ya registrada"
                                    if (serverMessage.contains("semana ya registrada")) {
                                        // Llama a la función updatePagoIdInSQLite
                                        updatePagoIdInSQLite(context, prestamoIdLocalOK, "", db);
                                        Log.i("PrestamoWorker", "Función updatePagoIdInSQLite ejecutada debido al mensaje del servidor.");
                                    }
                                }

                            }
                        } catch (Exception e) {
                            Log.e("PrestamoWorker", "Error al procesar el mensaje de error", e);
                        }

                        // Llamar al callback para notificar que esta tarea ha terminado
                        if (onComplete != null) onComplete.run();
                    });
                }

        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void updatePagoIdInSQLite(Context context, String pagoIdServidor, String idServidor, SQLiteDatabase db) {
        // Crear un ContentValues para actualizar el préstamo local con el `id` asignado por el servidor
        ContentValues values = new ContentValues();
        //values.put("id", idServidor);  // Actualiza el `prestamo_id` con el ID del servidor
        values.put("sinc", 0);

        // Actualizar el pago en SQLite con el `id` del servidor
        int rowsUpdatedPrestamos = db.update("pagos", values, "id = ?", new String[]{pagoIdServidor});

        if (rowsUpdatedPrestamos > 0) {
            Log.d("SQLite", "Prestamo actualizado correctamente con el ID del servidor: ");
        } else {
            Log.e("SQLite", "No se pudo actualizar el préstamo en SQLite.");
        }
    }
    private void checkForUpdates(Context context, SQLiteDatabase db) {
        Log.d("PagosWorker", "Checking for updates Pagos...");

        if (context == null) {
            Log.e("PagosWorker", "Context is null, cannot proceed with checkForUpdates.");
            return;
        }

        // Obtener préstamos pendientes con status = 8
        List<Map<String, String>> pagosPendientes = obtenerPagosPendientes(context, db);
        if (!pagosPendientes.isEmpty()) {
            Log.d("PrestamosWorker", "Encontrados " + pagosPendientes.size() + " pago pendientes para enviar al servidor.");

            // Crear un contador para las tareas pendientes
            AtomicInteger tareasPendientes = new AtomicInteger(pagosPendientes.size());

            // Enviar cada préstamo pendiente al servidor
            for (Map<String, String> pago : pagosPendientes) {
                insertDatosLocalAlServidor(context, pago, db, () -> {
                    // Reducir el contador cuando una tarea termine
                    int restantes = tareasPendientes.decrementAndGet();
                    Log.d("PrestamosWorker", "Tareas pendientes: " + restantes);

                    // Cuando todas las tareas hayan terminado, continuar con el resto del código
                    if (restantes == 0) {
                        continuarConElRestante(context, db);
                    }
                });
            }
        } else {
            Log.d("PrestamosWorker", "No hay préstamos pendientes de enviar al servidor.");
            continuarConElRestante(context, db);
        }
    }

    private void continuarConElRestante(Context context, SQLiteDatabase db) {
        Gson gson = new Gson();
        DatabaseHelper dbHelper = new DatabaseHelper(context);
        // Obtener los últimos 100 pagos desde SQLite
        List<Pago> localPagos = dbHelper.getLast100PagosFromSQLite();
        // Construcción del objeto JSON para la solicitud
        JSONObject data = new JSONObject();
        try {
            data.put("func", "index_app_PagosUpd");
            // Convertir allPagos a JSONArray
            JSONArray pagosArray = new JSONArray();
            for (Integer pagoId : allPagos) {
                pagosArray.put(pagoId);
            }

            // Añadir la lista de pagos al objeto JSON
            data.put("PrestamosId", pagosArray);

            Log.d("PAGOS", "Parametros enviados WORKER Pagos: " + data);

        } catch (JSONException e) {
            e.printStackTrace();
            return; // En caso de error, salir del método
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, urlPagos, data,
                response -> {
                    try {
                        // Log.d("PAGOS", "Respuesta recibida: " + response.toString());
                        JSONArray data1 = response.getJSONArray("data");
                        List<Pago> newPagos = new ArrayList<>();

                        // Procesar cada objeto JSON en la respuesta
                        for (int i = 0; i < data1.length(); i++) {
                            JSONObject obj = data1.getJSONObject(i);
                            Pago pago = gson.fromJson(obj.toString(), Pago.class);
                            newPagos.add(pago);
                        }
                        new SyncTask(context, newPagos, localPagos, db).execute();
                        // updatePagoInSQLite(context, newPagos, localPagos, db);

                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }, error -> {
            Log.e("PagosWorker", "Error en la solicitud de actualización de datos de Pagos: " + error.toString());
        });

        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }


    private void updatePagoInSQLite(Context context, List<Pago> serverPagos, List<Pago> localPagos, SQLiteDatabase db) {
        Log.d("Sync", "Actualizando el pago updatePagoInSQLite.............................");
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }
        // Crear un mapa para facilitar la búsqueda por ID en la lista del servidor
        Map<String, Pago> serverPagosMap = new HashMap<>();
        for (Pago serverPago : serverPagos) {
            serverPagosMap.put(serverPago.getId(), serverPago);
        }

        AtomicInteger remainingComparisons = new AtomicInteger(serverPagos.size());
        // Sincronizar pagos desde el servidor a la base de datos local
        for (Pago serverPago : serverPagos) {
            //Log.d("Sync", "Contenido del Pago del Servidor: " + serverPago.toString());
            Pago localPago = null;
            for (Pago pago : localPagos) {
                if (pago.getId().equals(serverPago.getId())) {
                    localPago = pago;
                    break;
                }
            }

            if (localPago != null) {
                String localUpdatedAt = localPago.getUpdated_at();

                if (localUpdatedAt == null || localUpdatedAt.isEmpty()) {
                    if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                        LocalDateTime now = LocalDateTime.now(); // Hora actual
                        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                        localUpdatedAt = now.format(formatter);
                    } else {
                        // Manejo para versiones antiguas (opcional, pero recomendable si soportas < API 26)
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                        localUpdatedAt = sdf.format(new Date()); // Hora actual sin modificaciones
                    }
                    // Insertar la hora actual en la base de datos SQLite
                    ContentValues values = new ContentValues();
                    values.put("updated_at", localUpdatedAt);

                    int rowsUpdated = db.update("pagos", values, "id = ?", new String[]{localPago.getId()});

                    if (rowsUpdated > 0) {
                        Log.d("SQLite", "Updated 'updated_at' for pagos ID: " + localPago.getId());
                    } else {
                        Log.e("SQLite", "Failed to update 'updated_at' for pagos ID: " + localPago.getId());
                    }
                }
                if (localUpdatedAt.isEmpty() || isNewer(serverPago.getUpdated_at(), localUpdatedAt)) {

                    updateValuesInDatabase(db, serverPago, remainingComparisons);

                    //Log.d("PagosWorker", "Contador después de actualizar: " + remainingComparisons.get());
                } else if (isNewer(localUpdatedAt, serverPago.getUpdated_at())) {
                    // Validar si los datos locales necesitan actualizarse en el servidor
                    if (localPago.getCantidad_normal_pagada() != serverPago.getCantidad_normal_pagada() &&
                            localPago.getCantidad_multa() != serverPago.getCantidad_multa()) {
                        enviarDatosLocalAlServidor(context, localPago, db, remainingComparisons);

                        //Log.d("PagosWorker", "Contador después de enviarDatosLocalAlServidor: " + remainingComparisons.get());
                    } else {
                        remainingComparisons.decrementAndGet();
                        //Log.d("PagosWorker", "Contador después de coinciden: " + remainingComparisons.get());
                    }
                } else {
                    //Log.d("Sync", "Las fechas son iguales. No se requiere ninguna acción para ID = " + serverPago.getId());
                    remainingComparisons.decrementAndGet();
                    //Log.d("PagosWorker", "Contador después de requiere ninguna acción: " + remainingComparisons.get());
                }
            } else {
                // Verificar si ya existe un pago con el mismo prestamo_id y semana
                boolean existePagoConMismaSemana = false;

                for (Pago localPagoCheck : localPagos) {

                    // Verificar si ya existe un pago con el mismo prestamo_id y semana
                    if (localPagoCheck.getPrestamo_id().equals(serverPago.getPrestamo_id()) &&
                            localPagoCheck.getSemana().equals(serverPago.getSemana())) {
                        existePagoConMismaSemana = true;
                        break; // Rompe el ciclo si se encuentra un pago existente
                    }
                }

                if (!existePagoConMismaSemana) {
                    // Log para confirmar que no existe un pago duplicado
                    insertPagosIntoSQLite(context, Collections.singletonList(serverPago), db, remainingComparisons);

                } else {
                    // Log adicional cuando se omite la inserción por duplicado

                    // Verificar y actualizar el pago existente con los valores del servidor
                    for (Pago localPagoCheck : localPagos) {
                        // Verificar si el prestamo_id y semana coinciden
                        if (localPagoCheck.getPrestamo_id().equals(serverPago.getPrestamo_id()) &&
                                localPagoCheck.getSemana().equals(serverPago.getSemana())) {

                            // Aquí puedes actualizar los valores del pago local con los valores del pago del servidor
                            localPagoCheck.setId(serverPago.getId());

                            // Llamar a un método que actualice el pago en la base de datos local
                            updatePagoInSQLite(db, localPagoCheck);  // Suponiendo que tienes un método para actualizar el pago

                            remainingComparisons.decrementAndGet();
                            break; // Una vez encontrado y actualizado el pago, podemos salir del bucle
                        }
                    }
                }
            }
        }
        Log.d("Sync", "Actualizando remainingComparisons............................." + remainingComparisons);
    }
    private void updatePagoInSQLite(SQLiteDatabase db, Pago pago) {
        ContentValues contentValues = new ContentValues();
        contentValues.put("id", pago.getId());  // Solo actualizamos el ID con el del servidor

        // Ejecutamos la actualización en la base de datos local, buscando el registro con el mismo prestamo_id y semana
        db.update("Pagos", contentValues, "prestamo_id = ? AND semana = ?",
                new String[]{pago.getPrestamo_id(), pago.getSemana()});
    }

    private void enviarDatosLocalAlServidor(Context context, Pago localPago, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        Log.d("PagoWorker", "Sending PAGOS local data to the server...");

        Handler mainHandler = new Handler(Looper.getMainLooper());

        if (context == null) {
            Log.e("PagosWorker", "Context is null. Cannot send data to server.");
            return; // Salir si el contexto es nulo
        }

        JSONObject data = new JSONObject();
        try {
            data.put("func", "updateFromLocal");
            data.put("prestamo_id", localPago.getPrestamo_id());
            data.put("semana", localPago.getSemana());
            data.put("tipo_pago", localPago.getTipo_pago());
            data.put("statusPago", localPago.getStatus());
            data.put("empleado_id", localPago.getEmpleado_id());
            data.put("concepto", localPago.getConcepto());
            data.put("updated_at", localPago.getUpdated_at());
            data.put("fecha_pago_realizada", localPago.getFecha_pago_realizada());
            data.put("cantidad_normal_pagada", localPago.getCantidad_normal_pagada());
            data.put("cantidad_multa", localPago.getCantidad_multa());
            data.put("folio", localPago.getFolio());
            remainingComparisons.decrementAndGet();
            //Log.d("PRESTSMOS", "Parametros enviados al servidor para actualizar: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            return;  // Si hay un error en la creación del JSON, salir
        }

        StringRequest stringRequest = new StringRequest(Request.Method.POST, urlPagos,
                new Response.Listener<String>() {
                    @Override
                    public void onResponse(String response) {
                        // Maneja la respuesta exitosa
                        Log.d("PagoWorker", "Respuesta del servidor: " + response);
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        Log.e("Volley Error", "Error occurred while sending request: " + error.toString());
                        if (error.networkResponse != null) {
                            Log.e("Volley Error", "Código de estado: " + error.networkResponse.statusCode);
                            String errorMessage = new String(error.networkResponse.data);
                            Log.e("Volley Error", "Respuesta del servidor enviarDatosLocalAlServidor con id:" + localPago.getPrestamo_id() +" SEMANA: "+ localPago.getSemana() +" ERROR: "+ errorMessage);
                        }
                    }
                }) {
            @Override
            public byte[] getBody() {
                return data.toString().getBytes(StandardCharsets.UTF_8);
            }

            @Override
            public String getBodyContentType() {
                return "application/json; charset=utf-8";
            }
        };
        // Agregar la solicitud a la cola de Volley
        RequestQueue requestQueue = Volley.newRequestQueue(context);
        requestQueue.add(stringRequest);
        isTaskPagosCompleted = true;
    }

    private void updateValuesInDatabase(SQLiteDatabase db, Pago pago, AtomicInteger remainingComparisons) {
        ContentValues values = new ContentValues();
        values.put("id", pago.getId());
        values.put("prestamo_id", pago.getPrestamo_id());
        values.put("cantidad_esperada_pago", pago.getCantidad_esperada_pago() );
        values.put("cantidad_normal_pagada", pago.getCantidad_normal_pagada());
        values.put("cantidad_multa", pago.getCantidad_multa());
        values.put("cantidad_pendiente", pago.getCantidad_pendiente());
        values.put("cantidad_total_pagada", pago.getCantidad_total_pagada());
        values.put("concepto", pago.getConcepto());
        values.put("fecha_pago", pago.getFecha_pago());
        values.put("fecha_pago_realizada", pago.getFecha_pago_realizada());
        values.put("folio", pago.getFolio());
        values.put("semana", pago.getSemana());
        values.put("balance", pago.getBalance());
        values.put("empleado_id", pago.getEmpleado_id());
        values.put("status", pago.getStatus());
        values.put("updated_at", pago.getUpdated_at());
        values.put("tipo_pago", pago.getTipo_pago());
        values.put("recuperado", pago.getRecuperado());
        // Actualizar el registro en la base de datos
        int rowsUpdated = db.update("pagos", values, "id=?", new String[]{String.valueOf(pago.getId())});
        remainingComparisons.decrementAndGet();
        if (rowsUpdated > 0) {
            Log.d("SQLite", "Registro actualizado correctamente pagos: ID = " + pago.getId());
        } else {
            Log.e("SQLite", "Error al actualizar el registro. ID no encontrado o fallo en la base de datos.");
        }
    }

    private boolean isNewer(String serverDate, String localDate) {
        // Verificar si alguna de las fechas es nula o vacía
        if (serverDate == null || localDate == null || serverDate.isEmpty() || localDate.isEmpty()) {
            Log.e("isNewer", "Una de las fechas es nula o vacía: serverDate=" + serverDate + ", localDate=" + localDate);
            return false; // O decide cómo manejar este caso
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX")); // Ajusta el formato según tu necesidad
        try {
            Date server = sdf.parse(serverDate);
            Date local = sdf.parse(localDate);

            return server != null && local != null && server.after(local);
        } catch (ParseException e) {
            e.printStackTrace();
            return false; // En caso de error, asumir que no es más reciente
        }
    }

    @SuppressLint("Range")
    private List<Map<String, String>> obtenerPagosPendientes(Context context, SQLiteDatabase db) {
        List<Map<String, String>> pagosPendientes = new ArrayList<>();

        String query = "SELECT * FROM pagos WHERE sinc = 1";
        Cursor cursor = db.rawQuery(query, null);

        if (cursor != null && cursor.moveToFirst()) {
            do {
                Map<String, String> pago = new HashMap<>();
                pago.put("id", cursor.getString(cursor.getColumnIndex("id")));
                pago.put("prestamo_id", cursor.getString(cursor.getColumnIndex("prestamo_id")));
                pago.put("cantidad_esperada_pago", cursor.getString(cursor.getColumnIndex("cantidad_esperada_pago")));
                pago.put("cantidad_normal_pagada", cursor.getString(cursor.getColumnIndex("cantidad_normal_pagada")));
                pago.put("cantidad_multa", cursor.getString(cursor.getColumnIndex("cantidad_multa")));
                pago.put("cantidad_pendiente", cursor.getString(cursor.getColumnIndex("cantidad_pendiente")));
                pago.put("cantidad_total_pagada", cursor.getString(cursor.getColumnIndex("cantidad_total_pagada")));
                pago.put("concepto", cursor.getString(cursor.getColumnIndex("concepto")));
                pago.put("fecha_pago", cursor.getString(cursor.getColumnIndex("fecha_pago")));
                pago.put("fecha_pago_realizada", cursor.getString(cursor.getColumnIndex("fecha_pago_realizada")));
                pago.put("folio", cursor.getString(cursor.getColumnIndex("folio")));
                pago.put("semana", cursor.getString(cursor.getColumnIndex("semana")));
                pago.put("balance", cursor.getString(cursor.getColumnIndex("balance")));
                pago.put("empleado_id", cursor.getString(cursor.getColumnIndex("empleado_id")));
                pago.put("status", cursor.getString(cursor.getColumnIndex("status")));
                pago.put("updated_at", cursor.getString(cursor.getColumnIndex("updated_at")));
                pago.put("tipo_pago", cursor.getString(cursor.getColumnIndex("tipo_pago")));
                pago.put("recuperado", cursor.getString(cursor.getColumnIndex("recuperado")));
                pago.put("sinc", cursor.getString(cursor.getColumnIndex("sinc")));

                pagosPendientes.add(pago);
            } while (cursor.moveToNext());
        }

        return pagosPendientes;
    }
}

